/*
 * Created on Apr 29, 2005 
 *
 */
package com.punosmusic.pitchdistributor;

/**
 * PitchNote represents a pitch in a scale. It has a repulsive force tpo other pitches close by,
 * inversely proportional to distance. We want to distribute pitches as evenly spaced as possible. A
 * PitchNode can transpose itself up or down by 1200 cents if the repulsive force is higher than a
 * threshold.
 * 
 * For Mike Winter
 * 
 * @author Nick Didkovsky, didkovn@mail.rockefeller.edu
 *  
 */
public class PitchNode implements Cloneable {

    private int pitch; // cents
    private double force;
    /** Arbitrarily chosen. If accumulated force exceeds repulsive force of a qtr tone, transpose */
    private static double transpositionForceThreshold = 4;
    private static int maxPitch = 2400;
    private static double minPitch = 1200;
    private static double K = 1E+4;

    // K arbitrarily chosen so pitches 100 cents apart repel with force = 1
    // pitches 1 cent apart repel with force = K or 10000

    public PitchNode(int pitch) {
        this.pitch = pitch;
    }

    /** If repulsive force exceeds threshold, transpose in direction of the force */
    public boolean evaluateState() {
        boolean stateChanged = false;
        if (Math.abs(getForce()) > getTranspositionForceThreshold()) {
            if (getForce() < 0 && pitch - 1200 >= minPitch) {
                pitch -= 1200;
                stateChanged = true;
            } else if (getForce() > 0 && pitch + 1200 <= maxPitch) {
                pitch += 1200;
                stateChanged = true;
            }
        }
        return stateChanged;
    }

    /**
     * @return Returns the transpositionForceThreshold.
     */
    public static double getTranspositionForceThreshold() {
        return transpositionForceThreshold;
    }

    /**
     * @param transpositionForceThreshold
     *            The transpositionForceThreshold to set.
     */
    public static void setTranspositionForceThreshold(double transpositionForceThreshold) {
        PitchNode.transpositionForceThreshold = transpositionForceThreshold;
    }

    /**
     * @return Returns the force.
     */
    public double getForce() {
        return force;
    }

    /**
     * @param force
     *            The force to set.
     */
    public void setForce(double force) {
        this.force = force;
    }

    /**
     * Simplified electrostatic force formula F= (k * q1 * q2) / r^2 where k is a constant, q1, and
     * q2 are magnitude of charges of two points. Here we just call k=1 and all pitches have the
     * same "charge" = 1
     */
    private static double calcForce(int p1, int p2) {
        double distance = p1 - p2;
        double sign = p1 > p2 ? 1 : -1;
        return sign * K / (distance * distance);
    }

    /** Calculate the repulsive force between this PitchNode and the Pitchnode passed in argument */
    private double calcForce(PitchNode p) {
        return calcForce(getPitch(), p.getPitch());
    }

    public void accumulateForce(PitchNode p) {
        if (this == p)
            return;
        setForce(getForce() + calcForce(p));
    }

    public void resetForce() {
        setForce(0);
    }

    public int getPitch() {
        return pitch;
    }

    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append("Pitch=" + pitch + ", force=" + getForce());
        return buf.toString();
    }

    public PitchNode getClone() {
        PitchNode clone = null;
        try {
            clone = (PitchNode) clone();
        } catch (CloneNotSupportedException e) {
            e.printStackTrace();
        }
        return clone;
    }

    public static void main(String[] args) {
        PitchNode p1 = new PitchNode(1200);
        PitchNode p2 = new PitchNode(1250);
        p1.resetForce();
        p2.resetForce();
        p1.accumulateForce(p2);
        p2.accumulateForce(p1);
        System.out.println(p1.toString());
        System.out.println(p2.toString());
    }
    /**
     * @return Returns the maxPitch.
     */
    public static int getMaxPitch() {
        return maxPitch;
    }
    /**
     * @param maxPitch The maxPitch to set.
     */
    public static void setMaxPitch(int maxPitch) {
        PitchNode.maxPitch = maxPitch;
    }
    /**
     * @return Returns the minPitch.
     */
    public static double getMinPitch() {
        return minPitch;
    }
    /**
     * @param minPitch The minPitch to set.
     */
    public static void setMinPitch(double minPitch) {
        PitchNode.minPitch = minPitch;
    }
}
