/*
 * Created on Apr 29, 2005 
 *
 */
package com.punosmusic.pitchdistributor;

import java.util.Enumeration;
import java.util.Vector;

/**
 * @author Nick Didkovsky, didkovn@mail.rockefeller.edu
 *  
 */
public class PitchNodeSystem {

    Vector pitchNodes = new Vector();

    public void addPitchNode(PitchNode p) {
        pitchNodes.addElement(p);
    }

    public void init() {
        reset();
        calcForces();
    }

    /**
     * Reset repulsive forces on all PitchNodes. Enumerate all PitchNode combinations and calculate
     * repulsive force between them. Finally, enumerate again and let PitchNodes transpose
     * themselves out of each others' way. You might run this repeatedly until it return false (ie
     * no PitchNode changed state), or better, set a limit on how many times you iterate this in
     * case you get into a perpetually flip-flopping state or some other interesting cycles.
     */
    public boolean gen() {
        boolean stateChanged = evaluateState();
        reset();
        calcForces();
        return stateChanged;
    }

    private boolean evaluateState() {
        boolean stateChanged = false;
        for (Enumeration e = pitchNodes.elements(); e.hasMoreElements();) {
            PitchNode pn = (PitchNode) e.nextElement();
            boolean transposed = pn.evaluateState();
            if (transposed) {
                stateChanged = true;
            }
        }
        return stateChanged;
    }

    private void calcForces() {
        for (int i = 0; i < pitchNodes.size(); i++) {
            for (int j = 0; j < pitchNodes.size(); j++) {
                if (i != j) {
                    PitchNode p1 = (PitchNode) pitchNodes.elementAt(i);
                    PitchNode p2 = (PitchNode) pitchNodes.elementAt(j);
                    p1.accumulateForce(p2);
                    p2.accumulateForce(p1);
                }
            }
        }
    }

    private void reset() {
        for (Enumeration e = pitchNodes.elements(); e.hasMoreElements();) {
            PitchNode pn = (PitchNode) e.nextElement();
            pn.resetForce();
        }
    }

    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append("PitchNodeSystem, size=" + pitchNodes.size() + "\n");
        for (Enumeration e = pitchNodes.elements(); e.hasMoreElements();) {
            PitchNode pn = (PitchNode) e.nextElement();
            buf.append(pn.toString() + "\n");
        }
        return buf.toString();
    }

    public double getTotalForce() {
        double accum = 0;
        for (Enumeration e=pitchNodes.elements(); e.hasMoreElements(); ) {
            PitchNode pn = (PitchNode)e.nextElement();
            accum += pn.getForce();
        }
        return accum;
    }
    public int size() {
        return pitchNodes.size();
    }
    
    public void clear() {
        pitchNodes.removeAllElements();
    }

    public PitchNode getPitchNode(int i) {
        return (PitchNode) pitchNodes.elementAt(i);
    }

    public static void main(String[] args) {

        PitchNode.setMinPitch(1200 - 600);
        PitchNode.setMaxPitch(2400 + 600);
        PitchNode.setTranspositionForceThreshold(4);

        PitchNodeSystem pitchNodeSystem = new PitchNodeSystem();
        pitchNodeSystem.addPitchNode(new PitchNode(1300));
        pitchNodeSystem.addPitchNode(new PitchNode(1302));

        pitchNodeSystem.init();	// !!! calcs initial repulsive forces, after this you may call gen() repeatedly

        System.out.println("PitchNodeSystems initial state\n" + pitchNodeSystem.toString());
        for (int i = 0; i < 4; i++) {
            boolean stateChanged = pitchNodeSystem.gen();
            System.out.println("Changed? " + stateChanged + "\n" + pitchNodeSystem.toString());
        }
    }

}
